/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Text;
using System.Collections;
using System.Collections.Specialized;
using System.Data;
using System.Runtime.Serialization;
using System.Diagnostics;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Logging;

namespace Borland.Eco.Persistence.Connection
{
	[Serializable()]
	public class DatabaseOperationFailedException: Exception
	{
#if !CF
		protected DatabaseOperationFailedException(SerializationInfo info, StreamingContext context): base(info, context){}
#endif
		public DatabaseOperationFailedException(string message, Exception innerException): base(message, innerException) {}
	}

	public abstract class AbstractIDatabaseImpl: IDatabase
	{
		protected AbstractIDatabaseImpl(SqlDatabaseConfig config)
		{
			m_Config = config;
		}
		private SqlDatabaseConfig m_Config;
		public SqlDatabaseConfig Config { get { return m_Config; } }
		internal int m_ExecedQueries;
		internal int m_ClosedQueries;
		internal IDbTransaction m_ImplicitTransaction;
		internal int m_OpenedQueries;
		private IDbTransaction m_CurrentTransaction;
		private int m_AllocatedQueries;
		private int m_ReturnedQueries;
		private int m_StatedTransactions;
		private int m_CommitedTransactions;
		private int m_RolledBackTransactions;
		private DateTime m_OpenSince;
		private bool m_ConnectionDisposed;

		#region IDatabase implementation

		///<exception cref="InvalidOperationException">Thrown if the database is not connected.</exception>
		protected void EnsureConnected()
		{
			if (!Connected)
				throw new InvalidOperationException(PersistenceStringRes.sNotConnected);
		}
		///<exception cref="InvalidOperationException">Thrown if there are active allocated queries.</exception>
		///<exception cref="InvalidOperationException">Thrown if there is an implicit or implicit transaction already started.</exception>
		void IDatabase.StartTransaction()
		{
			EnsureConnected();
			if (m_AllocatedQueries > m_ReturnedQueries)
				throw new InvalidOperationException(PersistenceStringRes.sAllocatedQueriesNotAllowedWhenStartingTransaction);
			if (m_ImplicitTransaction != null)
				throw new InvalidOperationException(PersistenceStringRes.sImplicitReadTransactionInProgress);
			if (m_CurrentTransaction != null)
				throw new InvalidOperationException(PersistenceStringRes.sTransactionInProgress);
			if (Config.UpdateIsolationLevel != IsolationLevel.Unspecified)
				m_CurrentTransaction = DbConnection.BeginTransaction(Config.UpdateIsolationLevel);
			else
				m_CurrentTransaction = DbConnection.BeginTransaction();
			m_StatedTransactions++;
		}

		///<exception cref="InvalidOperationException">Thrown if there was no current transaction to commit.</exception>
		void IDatabase.Commit()
		{
			EnsureConnected();
			if (m_CurrentTransaction == null)
				throw new InvalidOperationException(PersistenceStringRes.sNoTransactionToCommit);
			m_CurrentTransaction.Commit();
			m_CurrentTransaction = null;
			m_CommitedTransactions++;
		}

		///<exception cref="InvalidOperationException">Thrown if there was no current transaction to roll back.</exception>
		void IDatabase.RollBack()
		{
			EnsureConnected();
			if (m_CurrentTransaction == null)
				throw new InvalidOperationException(PersistenceStringRes.sNoTransactionToRollback);
			m_CurrentTransaction.Rollback();
			m_CurrentTransaction = null;
			m_RolledBackTransactions++;
		}

		void IDatabase.Open()
		{
			if (Connected)
				return;
			DbConnection.Open();
			m_OpenSince = DateTime.Now;
		}

		DateTime IDatabase.OpenSince
		{
			get { return m_OpenSince; }
		}

		///<exception cref="InvalidOperationException">Thrown if there were any problems closing the connection.</exception>
		public void Close()
		{
			if (!Connected)
				return;
			string msg = "";
			if (m_AllocatedQueries != m_ReturnedQueries)
				msg = msg + String.Format(CultureInfo.InvariantCulture, "Allocated queries: {0} returned: {1} ", m_AllocatedQueries, m_ReturnedQueries); // do not localize

			if (m_OpenedQueries != m_ClosedQueries)
				msg = msg + String.Format(CultureInfo.InvariantCulture, "Opened queries: {0} Closed: {1}" , m_OpenedQueries, m_ClosedQueries); // do not localize

			if (m_ImplicitTransaction != null)
			{
				m_ImplicitTransaction.Commit();
				m_ImplicitTransaction = null;
				msg = msg + "Uncommited implicit transactions active" ; // do not localize
			};

			DbConnection.Close();
			m_AllocatedQueries = 0;
			m_ReturnedQueries = 0;
			m_OpenedQueries = 0;
			m_ExecedQueries = 0;
			m_ClosedQueries = 0;
			if (msg.Length > 0)
				throw new InvalidOperationException(msg);
		}

		public IQuery GetQuery()
		{
			EnsureConnected();
			m_AllocatedQueries++;
			return CreateCommand();
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="query"/> is null</exception>
		public void ReleaseQuery(IQuery query)
		{
			if (query == null) throw new ArgumentNullException("query"); // do not localize
			EnsureConnected();
			((AbstractIQueryImpl)query).Dispose();
			m_ReturnedQueries++;
		}

		public void DisposeConnection()
		{
			if (m_ConnectionDisposed)
				return;
			if (Connected)
				Close();
			DbConnection.Dispose();
			m_ConnectionDisposed = true;
		}

		IExecQuery IDatabase.GetExecQuery()
		{
			EnsureConnected();
			m_AllocatedQueries++;
			return CreateCommand();
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="query"/> is null</exception>
		void IDatabase.ReleaseExecQuery(IExecQuery query)
		{
			if (query == null) throw new ArgumentNullException("query"); // do not localize
			EnsureConnected();
			((AbstractIQueryImpl)query).Dispose();
			m_ReturnedQueries++;
		}

		///<exception cref="NotImplementedException">Thrown always.</exception>
		void IDatabase.ReleaseCachedObjects()
		{
			throw new NotImplementedException();
		}

		public abstract StringCollection AllTableNames(string pattern, bool systemTables);

		bool IDatabase.TableExists(string tableName)
		{
			return AllTableNames("", false).Contains(tableName);
		}

		public abstract DataTable GetIndexDefsForTable(string tableName);

		public bool Connected // Doubles for IDatabase.Connected
		{
			get
			{
				if (m_ConnectionDisposed)
					return false;
				return DbConnection.State == System.Data.ConnectionState.Open;
			}
		}
		bool IDatabase.InTransaction
		{
			get { return m_CurrentTransaction != null; }
		}
		public abstract IDbConnection DbConnection { get; }

		#endregion

		protected abstract AbstractIQueryImpl CreateCommand();

		///<exception cref="ArgumentNullException">Thrown if <paramref name="text"/> is null</exception>
		protected static bool FindFirstParam(string text, out string head, out string name, out string tail)
		{
			if (text == null)
				throw new ArgumentNullException("text");
			int start = text.IndexOf(':');
			if (start == -1)
			{
				head = "";
				name = "";
				tail = text;
				return false;
			}
			else
			{
				int stop = start+1;
				int l = text.Length;
				while ((stop < l)
					&& (Char.IsLetterOrDigit(text[stop]) || text[stop] == '_'))
				stop++;
				head = text.Substring(0, start);
				name = text.Substring(start + 1, stop - start - 1);
				tail = text.Substring(stop);
				return true;
			}
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="parameters"/> is null</exception>
		protected static string ConvertParametersToLiterals(string commandText,  IDataParameterCollection parameters)
		{
			if (parameters == null)
				throw new ArgumentNullException("parameters");
			string tail = commandText;
			string head;
			string paramName;
			string paramValue;
			StringBuilder sb = new StringBuilder();
			while (FindFirstParam(tail, out head, out paramName, out tail))
			{
				IDataParameter param = (IDataParameter)parameters[paramName];
				// FIXME mapper should have AsParamString
				if ((param.DbType == DbType.AnsiString) || (param.DbType == DbType.String))
					paramValue = "'" + param.Value.ToString() + "'";
				else if ((param.DbType == DbType.Time) || (param.DbType == DbType.Date) || (param.DbType == DbType.DateTime))
					paramValue = "'" + param.Value.ToString() + "'";
				else
					paramValue = param.Value.ToString();

				sb.Append(head);
				sb.Append(paramValue);
			}

			sb.Append(tail);
			return sb.ToString();
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="parameters"/> is null</exception>
		protected static string ConvertParametersToAtPrefix(string commandText,  IDataParameterCollection parameters)
		{
			if (parameters == null)
				throw new ArgumentNullException("parameters");
			string tail = commandText;
			string head;
			string paramName;
			StringBuilder result = new StringBuilder();
			while (FindFirstParam(tail, out head, out paramName, out tail))
			{
				IDataParameter param = (IDataParameter)parameters[paramName];
				param.ParameterName = "@" + param.ParameterName; // do not localize
				result.Append(head);
				result.Append("@");
				result.Append(paramName);
			}
			result.Append(tail);
			return result.ToString();
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="parameters"/> is null</exception>
		protected static string ConvertParametersToUpperCase(string commandText, IDataParameterCollection parameters)
		{
			if (parameters == null)
				throw new ArgumentNullException("parameters");
			string tail = commandText;
			string head;
			string paramName;
			StringBuilder result = new StringBuilder();
			while (FindFirstParam(tail, out head, out paramName, out tail))
			{
				IDataParameter param = (IDataParameter)parameters[paramName];
				param.ParameterName = param.ParameterName.ToUpper(CultureInfo.InvariantCulture);
				result.Append(head);
				result.Append(":");
				result.Append(paramName.ToUpper(CultureInfo.InvariantCulture));
			}
			result.Append(tail);
			return result.ToString();
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="parameters"/> is null</exception>
		protected static string ConvertParametersToQuestionMarks(string commandText, IDataParameterCollection parameters)
		{
			if (parameters == null)
				throw new ArgumentNullException("parameters");

			commandText = PreProcess(commandText);

			string tail = commandText;
			string head;
			string paramName;
			StringBuilder result = new StringBuilder();
			IList paramsAsList = (IList)parameters;
			int foundParams = 0;

			while (FindFirstParam(tail, out head, out paramName, out tail))
			{
				int paramIndex = parameters.IndexOf(paramName);
				if (paramIndex == -1)
					throw new InvalidOperationException(String.Format(CultureInfo.InvariantCulture, "Unable to find parameter {0} in the paramlist", paramName)); // Do not localize

				if (paramIndex != foundParams)
				{
					if (paramIndex < foundParams)
						throw new InvalidOperationException(String.Format(CultureInfo.InvariantCulture, "Parameter reused {0}", paramName)); // Do not localize

					else
					{
						IDataParameter param = (IDataParameter)paramsAsList[paramIndex];
						paramsAsList.RemoveAt(paramIndex);
						parameters.Insert(foundParams, param);
					}
				}
				foundParams++;
				result.Append(head);
				result.Append("?");
			}

			result.Append(tail);

			string finalResult = PostProcess(result.ToString());

			return finalResult;
		}

		protected static string HIDE_WORD1 = ":new.id";
		protected static string HIDE_WORD_REPLACEMENT1 = "__new.id__";		

		protected static string PreProcess(string text)
		{
			return text.Replace(HIDE_WORD1, HIDE_WORD_REPLACEMENT1);				
		}

		protected static string PostProcess(string text)
		{
			return text.Replace(HIDE_WORD_REPLACEMENT1, HIDE_WORD1);
		}

		protected IDbTransaction CurrentTransaction
		{
			get { return m_CurrentTransaction; }
		}

		protected IDbTransaction ImplicitTransaction
		{
			get { return m_ImplicitTransaction; }
		}
		internal IDbTransaction StartImplicitTransaction()
		{
			EnsureConnected();
			if (m_ImplicitTransaction != null)
				throw new InvalidOperationException(PersistenceStringRes.sImplicitReadTransactionInProgress);
			if (m_CurrentTransaction != null)
				throw new InvalidOperationException(PersistenceStringRes.sTransactionInProgress);
			if (Config.FetchIsolationLevel != IsolationLevel.Unspecified)
				m_ImplicitTransaction = DbConnection.BeginTransaction(Config.FetchIsolationLevel);
			else
				m_ImplicitTransaction = DbConnection.BeginTransaction();
			return m_ImplicitTransaction;
		}

		internal void EndImplicitTransaction()
		{
			EnsureConnected();
			m_ImplicitTransaction.Commit();
			m_ImplicitTransaction = null;
		}

		protected internal abstract string HandleCommandParameters(string commandText, IDataParameterCollection parameters);

		public bool InTransaction
		{
			get { return m_CurrentTransaction != null; }
		}

		public bool InImplicitTransaction
		{
			get { return m_ImplicitTransaction != null; }
		}

		public virtual bool AlwaysReadInTransactions
		{
			get { return Config.FetchIsolationLevel != IsolationLevel.Unspecified; }
		}
	}

	public abstract class AbstractIQueryImpl: IQuery, IExecQuery
	{
		protected AbstractIQueryImpl() { }
		private int m_LastRowsAffected;
		private bool m_EndOfQuery;
		private IDbCommand m_DbCommand;
		private IDataReader m_DataReader;
		private IDataRecord m_DataRecord;
		private readonly AbstractIDatabaseImpl m_ConnectionAdapter;
		private string m_SQL;

		protected AbstractIQueryImpl(IDbCommand dbCommand, AbstractIDatabaseImpl connectionAdapter)
		{
			m_DbCommand = dbCommand;
			m_ConnectionAdapter = connectionAdapter;
		}

		private bool InImplicitTransaction
		{
			get { return ((m_ConnectionAdapter != null) && (m_ConnectionAdapter.InImplicitTransaction)); }
		}

		#region IDataSet implementation

		IField IDataSet.FieldByName(string fieldName)
		{
			return new DataRecordAdapter(DataRecord.GetOrdinal(fieldName), DataRecord);
		}

		protected virtual void InternalClose()
		{
			try
			{
				ReleaseReader();
				DbCommand.CommandText = "";
				DbCommand.Parameters.Clear();
			}
			finally
			{
				if (InImplicitTransaction)
					EndImplicitTransaction();
			}
		}

		public void Close()
		{
			m_ConnectionAdapter.m_ClosedQueries++;
			InternalClose();
		}

		public void Next()
		{
			m_EndOfQuery = !DataReader.Read();
		}

		void IDataSet.Open()
		{
			if (!m_ConnectionAdapter.InTransaction &&
				!m_ConnectionAdapter.InImplicitTransaction &&
				m_ConnectionAdapter.AlwaysReadInTransactions)
			{
				m_DbCommand.Transaction = m_ConnectionAdapter.StartImplicitTransaction();
			}
			try
			{
				CreateReader();
				Next();
				m_ConnectionAdapter.m_OpenedQueries++;
			}
			catch
			{
				if (InImplicitTransaction)
				  EndImplicitTransaction();
				throw;
			}
		}

		///<exception cref="ArgumentException">Thrown if <paramref name="distance"/> is negative.</exception>
		int IDataSet.MoveBy(int distance)
		{
			// This is very far from optimal, is is possible to improve on this?
			if (distance < 0)
				throw new ArgumentException(PersistenceStringRes.sNegativeDistanceNotAllowed, "distance"); // do not localize
			int result = 0;
			for (int i = 0; i < distance  && !Eof; i++)
			{
				Next();
				result++;
			}
			return result;
		}

		public bool Eof
		{
			get { return m_EndOfQuery; }
		}

		int IDataSet.FieldCount
		{
			get { return DataRecord.FieldCount; }
		}

		IField IDataSet.this[int index]
		{
			get { return new DataRecordAdapter(index, DataRecord); }
		}

		#endregion

		#region IParameterized implementation

		void IParameterized.ClearParams()
		{
			DbCommand.Parameters.Clear();
		}

		public virtual IDataParameter ParamByName(string value)
		{
			int index = DbCommand.Parameters.IndexOf(value);
			if (index != -1)
			{
				return (IDataParameter)DbCommand.Parameters[value];
			}
			else
				return CreateParam(value);
		}

		#endregion

		#region IQueryBase implementation

		void IQueryBase.AssignSqlText(string sql)
		{
			m_SQL = sql;
		}

		string IQueryBase.SqlText
		{
			get { return m_SQL; }
		}

		#endregion

		#region IExecQuery implementation

		public virtual void StartSqlBatch()
		{
			// Nothing yet
		}

		public virtual void EndSqlBatch()
		{
			// Nothing yet
		}

		public virtual void FailSqlBatch()
		{
			// Nothing yet
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="p"/> is null</exception>
		public virtual string GetParameterTypeDescription(IDataParameter p)
		{
			if (p == null) throw new ArgumentNullException("p"); // do not localize
			return "(DbType: " + p.DbType.ToString() + ")";
		}

		private void BuildParameterInfoString(IList paramList, StringBuilder sb)
		{
			foreach (IDataParameter p in paramList)
			{
				sb.Append(String.Format(CultureInfo.InvariantCulture, "\r\n {0}=", p.ParameterName)); // do not localize
				if (p.Value != null)
				{
					string paramValue = p.Value.ToString();
					if (paramValue != null && paramValue.Length > 30)
					{
						sb.Append(paramValue.Substring(1, 30));
						sb.Append(String.Format(CultureInfo.InvariantCulture, "... (length={0})", paramValue.Length)); // do not localize
					}
					else
					{
						sb.Append(p.Value.ToString());
					}
					sb.Append(" ("); // do not localize
					sb.Append(p.Value.GetType());
					sb.Append(")"); // do not localize

				}
				else
					sb.Append("null"); // do not localize

				sb.Append(" " + GetParameterTypeDescription(p));
			}
		}

		///<exception cref="DatabaseOperationFailedException">Thrown if the script execution failed. The original exception is made an inner exception.</exception>
		public virtual void ExecSql()
		{
			try
			{
				string newSql = m_ConnectionAdapter.HandleCommandParameters(m_SQL, DbCommand.Parameters);
#if !CF
				if (EcoLogSwitches.LogSql)
					Trace.WriteLine(newSql.Trim(), EcoTraceCategories.Sql);
#endif
				if (newSql != DbCommand.CommandText)
				{
					DbCommand.CommandText = newSql;
				}
				m_LastRowsAffected = DbCommand.ExecuteNonQuery();
				DbCommand.Parameters.Clear();
			}
			catch (Exception e)
			{
				IList PList = DbCommand.Parameters as IList;
				StringBuilder sb = new StringBuilder();
				sb.Append(PersistenceStringRes.sExecSqlError(DbCommand.CommandText, PList.Count));
				BuildParameterInfoString(PList, sb);
				sb.Append("\r\n"); // do not localize
				sb.Append(e);
				throw new DatabaseOperationFailedException(sb.ToString(), e);
			}
			m_ConnectionAdapter.m_ExecedQueries++;
		}

		int IExecQuery.RowsAffected
		{
			get { return m_LastRowsAffected; }
		}

		#endregion

		private bool IsOpen
		{
			get { return m_DataReader != null; }
		}

		protected IDbCommand DbCommand
		{
			get { return m_DbCommand; }
		}

		protected IDataRecord DataRecord
		{
			get { return m_DataRecord; }
		}

		protected IDataReader DataReader
		{
			get { return m_DataReader; }
		}

		protected void EndImplicitTransaction()
		{
			m_ConnectionAdapter.EndImplicitTransaction();
		}

		internal void Dispose()
		{
			InternalClose();
			m_DbCommand.Dispose();
			m_DbCommand = null;
		}

		protected virtual void CreateReader()
		{
			try
			{
				string newSql = m_ConnectionAdapter.HandleCommandParameters(m_SQL, DbCommand.Parameters);

#if !CF
				if (EcoLogSwitches.LogSql)
					Trace.WriteLine(newSql.Trim(), EcoTraceCategories.Sql);
#endif

				if (newSql != DbCommand.CommandText)
					DbCommand.CommandText = newSql;
			}
			catch (Exception e)
			{
				string msg = PersistenceStringRes.sErrorHandlingSql(m_SQL);
				msg = msg + "\r\n" + e.ToString(); // do not localize
				throw new DatabaseOperationFailedException(msg, e);
			}

			try
			{
				m_DataReader = DbCommand.ExecuteReader();
				m_DataRecord = (IDataRecord)m_DataReader;
			}
			catch (Exception e2)
			{
				string msg = PersistenceStringRes.sErrorExecutingSql(DbCommand.CommandText);
				msg = msg + "\r\n" + e2.Message; // do not localize
				throw new DatabaseOperationFailedException(msg, e2);
			}
		}

		protected void ReleaseReader()
		{
			if (m_DataReader != null)
				m_DataReader.Close();
			m_DataReader = null;
			m_DataRecord = null;
		}

		protected abstract IDataParameter CreateParam(string paramName);

		private class DataRecordAdapter: IField
		{
			private readonly int m_FieldIndex;
			private readonly IDataRecord m_DataRecord;

			public  DataRecordAdapter(int fieldIndex, IDataRecord dataRecord)
			{
				m_DataRecord = dataRecord;
				m_FieldIndex = fieldIndex;
			}

			bool IField.IsNull
			{
				get { return m_DataRecord.IsDBNull(m_FieldIndex); }
			}

			object IField.Value
			{
				get { return m_DataRecord.GetValue(m_FieldIndex); }
			}

			System.Int16 IField.AsInt16
			{
				get
				{
					if (m_DataRecord.GetFieldType(m_FieldIndex) == typeof(System.Decimal))
						return System.Decimal.ToInt16(m_DataRecord.GetDecimal(m_FieldIndex));
					else
						return m_DataRecord.GetInt16(m_FieldIndex);
				}
			}
			System.Int32 IField.AsInt32
			{
				get
				{
					if (m_DataRecord.GetFieldType(m_FieldIndex) == typeof(System.Decimal))
						return System.Decimal.ToInt32(m_DataRecord.GetDecimal(m_FieldIndex));
					else
						return m_DataRecord.GetInt32(m_FieldIndex);
				}
			}
		}
	}

}
